<?php

class Xoo_Helper{

	public $slug, $path, $helperArgs;
	public $admin;
	public $liccheck  			= false;
	public $registerLicenseUrl 	= 'https://xootix.com/wp-json/license/v1/register';
	public $licenseInfoURL 		= 'https://xootix.com/wp-json/license/v1/info';
	public $updateURL 			= 'https://xootix.com/wp-json/plugins/v1/update';
	public $cachedPluginInfo 	= false;
	public $licensePageURL;

	public function __construct( $slug, $path, $helperArgs = array() ){

		$helperArgsDefaults = array(
			'liccheck' 	=> 'no',
			'mainfile' 	=> ''
		);

		$this->helperArgs 		= wp_parse_args( $helperArgs, $helperArgsDefaults );
		$this->slug 			= $slug;
		$this->path 			= $path;
		$this->liccheck 		= $this->helperArgs['liccheck'] === 'no';
		$this->licensePageURL 	= $slug.'-license'; 

		$this->set_constants();
		$this->includes(); 
		$this->hooks();
	}


	public function set_constants(){
		$this->define( 'XOO_FW_URL', untrailingslashit(plugin_dir_url( XOO_FW_DIR .'/'.basename( XOO_FW_DIR ) ) ) );
		$this->define( 'XOO_FW_VERSION', '1.2' );
	}

	public function define( $name, $value ){
		if( !defined( $name ) ){
			define( $name, $value );
		}
	}

	public function includes(){
		require_once __DIR__.'/admin/class-xoo-admin-settings.php';
		$this->admin = new Xoo_Admin( $this );
	}


	public function hooks(){
		add_action( 'init', array( $this, 'internationalize' ) );
		add_action( 'admin_init', array( $this, 'time_to_update_theme_templates_data' ) );

		if( $this->liccheck ){
			add_action( 'wp_ajax_xoo_ff_license_register', array( $this, 'license_register_form_handle' ) );
			add_filter('site_transient_update_plugins', array( $this, 'check_for_plugin_update' ) );
			add_filter( 'plugins_api', array( $this, 'show_plugin_info' ), 20, 3 );
		}
	}


	public function get_template( $template_name, $args = array(), $template_path = '', $return = false ){

		$located = $this->locate_template( $template_name, $template_path );

		$located = apply_filters( 'xoo_'.$this->slug.'_get_template', $located, $template_name, $args, $template_path );

	    if ( $args && is_array ( $args ) ) {
	        extract ( $args );
	    }

	    if ( $return ) {
	        ob_start ();
	    }


	    // include file located
	    if ( file_exists ( $located ) ) {
	        include ( $located );
	    }

	    if ( $return ) {
	        return ob_get_clean ();
	    }
	}

	public function locate_template( $template_name, $template_path ){

		$lookIn = array(
			'templates/'.$this->slug.'/'.$template_name,
			'templates/'.$this->slug.'/'.basename( $template_name ),
			$template_name,
		);

		 // Look within passed path within the theme - this is priority.
		$template = locate_template( $lookIn );

		//Check woocommerce directory for older version
		if( !$template && class_exists( 'woocommerce' ) ){
			if( file_exists( WC()->plugin_path() . '/templates/' . $template_name ) ){
				$template = WC()->plugin_path() . '/templates/' . $template_name;
			}
		}


	    if ( ! $template ) {
	    	if( $template_path ){
	    		$template = $template_path.'/'.$template_name;
	    		
	    	}
	    	else{
	    		$template = $this->path .'/templates/'. $template_name;
	    	}
	    }

	    return apply_filters( 'xoo_'.$this->slug.'_template_located', $template, $template_name, $template_path );
	}


	public function get_option( $key, $subkey = '' ){
		$option = get_option( $key );
		if( $subkey ){
			if( !isset( $option[ $subkey ] ) ) return;
			return !is_array( $option[ $subkey ] ) ? esc_attr( $option[ $subkey ] ) : $option[ $subkey ];
		}
		else{
			return $option;
		}
	}


	public function internationalize(){
        load_plugin_textdomain( $this->slug, FALSE, basename( $this->path) . '/languages/' ); // Plugin Languages
	}


	/**
	 * Retrieve metadata from a file. Based on WP Core's get_file_data function.
	 */
	public function get_file_version( $file ) {

		// Avoid notices if file does not exist.
		if ( ! file_exists( $file ) ) {
			return '';
		}

		// We don't need to write to the file, so just open for reading.
		$fp = fopen( $file, 'r' ); // @codingStandardsIgnoreLine.

		// Pull only the first 8kiB of the file in.
		$file_data = fread( $fp, 8192 ); // @codingStandardsIgnoreLine.

		// PHP will close file handle, but we are good citizens.
		fclose( $fp ); // @codingStandardsIgnoreLine.

		// Make sure we catch CR-only line endings.
		$file_data = str_replace( "\r", "\n", $file_data );
		$version   = '';

		if ( preg_match( '/^[ \t\/*#@]*' . preg_quote( '@version', '/' ) . '(.*)$/mi', $file_data, $match ) && $match[1] ) {
			$version = _cleanup_header_comment( $match[1] );
		}

		return $version;
	}



	/**
	 * Look for theme templates
	 *
	 * @return array
	 */
	public function get_theme_templates( $scan_woocommerce = false ) {
		$override_data  = array();
		$template_paths = apply_filters( 'xoo_'.$this->slug.'_template_overrides_scan_paths', array( 'templates' => $this->path . '/templates/' ) );
		$scanned_files  = $theme_templates = array();

		foreach ( $template_paths as $lookInDir => $template_path ) {
			$scanned_files[ $lookInDir ] = $this->scan_template_files( $template_path );
		}

		foreach ( $scanned_files as $lookInDir => $files ) {
			foreach ( $files as $file ) {

				$basename = basename( $file );

				if ( file_exists( get_stylesheet_directory() . '/templates/' . $this->slug .'/'. $file ) ) {
					$theme_file = get_stylesheet_directory() . '/templates/' . $this->slug .'/'. $file;
				} elseif (  class_exists( 'woocommerce' ) && $scan_woocommerce && file_exists( get_template_directory() . '/' . WC()->template_path() . $file ) ) {
					$theme_file = get_template_directory() . '/' . WC()->template_path() . $file;
				} else {
					$theme_file = false;
				}


				if ( ! empty( $theme_file ) ) {
					$core_version  = $this->get_file_version( $template_paths[ $lookInDir ] .'/'. $file );
					$theme_version = $this->get_file_version( $theme_file );
					$theme_templates[] = array(
						'file' 			=> $theme_file,
						'name' 			=> str_replace( array( WP_CONTENT_DIR, '\\' ) , array( '', '/' ), $theme_file ),
						'theme_version' => $theme_version,
						'core_version' 	=> $core_version,
						'is_outdated' 	=> version_compare( $core_version , $theme_version, '>' ) ? 'yes' : 'no',
						'basename' 		=> $basename,
					);
				}
			}
		}

		return $theme_templates;
	}



	/**
	 * Scan the template files.
	 *
	 * @param  string $template_path Path to the template directory.
	 * @return array
	 */
	public function scan_template_files( $template_path ) {
		$files  = @scandir( $template_path ); // @codingStandardsIgnoreLine.
		$result = array();

		if ( ! empty( $files ) ) {

			foreach ( $files as $key => $value ) {

				if ( ! in_array( $value, array( '.', '..' ), true ) ) {

					if ( is_dir( $template_path . DIRECTORY_SEPARATOR . $value ) ) {
						$sub_files = $this->scan_template_files( $template_path . DIRECTORY_SEPARATOR . $value );
						foreach ( $sub_files as $sub_file ) {
							$result[] = $value . DIRECTORY_SEPARATOR . $sub_file;
						}
					} else {
						$result[] = $value;
					}
				}
			}
		}
		return $result;
	}



	public function get_outdated_section(){

		$odTempData = $this->get_theme_templates_data();
		ob_start();
		?>
		<div class="xoo-outdatedtemplates">
			<?php if( $odTempData['has_outdated'] === "yes" ): ?>
				<span>You're using outdated version of templates, please fetch a new copy from the plugin templates folder</span>
				<ul>
					<?php
					foreach ( $odTempData['templates'] as $template_data ){
						if( $template_data['is_outdated'] !== 'yes' ) continue;
						echo '<li><span class="dashicons dashicons-warning"></span>'. esc_html( $template_data['name'] ).'</li>';
					}
					?>
				</ul>
			<?php else: ?>
				<div>Templates Status
				<span class="dashicons dashicons-yes-alt" style="font-size: 14px;color: #008000;line-height: 1.3;"></span>
				<a href="https://docs.xootix.com/<?php echo esc_attr( $this->slug ); ?>" target="_blank">How to override?</a>
				</div>
			<?php endif; ?>
			<span>Last checked: <?php echo esc_html( get_date_from_gmt( date( 'Y-m-d H:i:s', $odTempData['last_scanned'] ) ) ); ?></span>
			<a href="<?php echo esc_url( add_query_arg( array( 'scan_templates' => 'yes' , 'slug' => $this->slug ) ) ); ?>">Check again</a>
		</div>
		<?php
		return ob_get_clean();
	}


	public function get_theme_templates_data(){

		$data = (array) get_option( 'xoo_'.$this->slug.'_theme_templates_data' );
		if( empty( $data ) || !isset( $data['last_scanned'] ) ){
			return $this->update_theme_templates_data();
		}
		return $data;
	}


	public function time_to_update_theme_templates_data(){

		$tempData = $this->get_theme_templates_data();

		if(  ( ( time() - $tempData['last_scanned'] ) > ( 86400 * 1 ) ) || ( isset( $_GET['scan_templates'] ) && isset( $_GET['slug'] ) && $_GET['slug'] === $this->slug ) ){
			$this->update_theme_templates_data();
			wp_safe_redirect( remove_query_arg( array( 'scan_templates', 'slug' ) ) );
			die();
		}
	}


	public function update_theme_templates_data(){

		$tempData = array();

		$theme_templates = (array) $this->get_theme_templates( true );

		$has_outdated = 'no';

		foreach ( $theme_templates as $template ) {
			if( $template['is_outdated'] === "yes" ){
				$has_outdated = "yes";
				break;
			}
		}

		$tempData['has_outdated'] 	= $has_outdated;
		$tempData['templates'] 		= $theme_templates;
		$tempData['last_scanned'] 	= time();

		update_option( 'xoo_'.$this->slug.'_theme_templates_data', $tempData );

		return $tempData;
	
	}

	public function box_shadow_desc($value){
		$html = '<a href="https://box-shadow.dev/" target="__blank">Preview & click on "Show code" -> copy value</a>';
		if( $value ){
			$html .= 'Default: '.$value;
		}
		return $html;
	}

	public function notice_html( $message, $notice_type = 'success' ){

		$classes = $notice_type === 'error' ? 'xoo-ff-notice-error' : 'xoo-ff-notice-success';
		
		$html = '<li class="'.$classes.'">'.$message.'</li>';
		
		return $html;

	}


	public function license_info_http_request(){

		$httpData = $this->get_license_info();

		$httprequest = wp_remote_post(
			$this->licenseInfoURL,
			array(
				'body' => $httpData
			)
		);

		$response = json_decode(wp_remote_retrieve_body($httprequest), true);

		update_option( 'xoo-license-info-'.$this->slug, isset( $response['license_key'] ) ? $response : '' );

		return $response;
	}


	public function license_register_form_handle(){

		if( !isset( $_POST['xoo_license_nonce_value'] ) || !wp_verify_nonce( $_POST['xoo_license_nonce_value'], 'xoo_license_nonce' ) ) return;

		try {

			$licenseKey = $_POST['xoo-license-key'];

			if( !$licenseKey ){
				throw new Xoo_Exception( 'License key required' );
			}

			$httpData = array(
				'license_key' 	=> $licenseKey,
				'plugin_slug' 	=> $_POST['xoo-plugin-slug'],
				'site_url' 		=> $_POST['xoo-license-website']
			);

			$httprequest = wp_remote_post(
				$this->registerLicenseUrl,
				array(
					'body' => $httpData
				)
			);

			$response = json_decode(wp_remote_retrieve_body($httprequest), true);

			if( $response ){

				if( isset( $response['error'] ) ){
					throw new Xoo_Exception( $response['message'] );
				}

				if( isset( $response['license_key'] ) ){ //license gets registered sucesfully.

					update_option( 'xoo-license-info-'.$this->slug, $response );

					$jsResponse = array(
						'message' => 'License registered successfully',
						'success' => 1
					);
					wp_send_json( $jsResponse );
				}
			}
			else{
				throw new Xoo_Exception( 'Something went wrong. Please contact support' );
			}

			

		} catch ( Xoo_Exception $e) {
			wp_send_json(array(
				'error' 	=> 1,
				'message' 	=> $e->getMessage()
			));
		}

		exit;
		
	}

	public function init_plugin(){
		return empty($this->get_license_info('license_key'));
	}

	public function get_license_info( $key = '' ){

		$info = get_option( 'xoo-license-info-'.$this->slug );

		if( $key ){
			return isset( $info[ $key ] ) ? $info[ $key ] : '';
		}

		return $info;
	}

	public function is_license_active(){
		return $this->get_license_info('is_active');
	}

	public function has_license_expired(){

		$expires_at = $this->get_license_info('expires_at');

		return $expires_at && new DateTime() > new DateTime($expires_at);

	}

	public function get_license_expiry_date(){
		$expiryDate = $this->get_license_info('expires_at');
		return $expiryDate ? (new DateTime( $expiryDate ) )->format('d-m-Y') : '';
	}


	public function plugin_update_info_http_request(){

		$httpData = $this->get_db_license_data();

		if( !$httpData ) return;

		$httprequest = wp_remote_post(
			$this->updateURL,
			array(
				'body' => $httpData
			)
		);

		$response = json_decode( wp_remote_retrieve_body($httprequest), true);

		if( !$response ){ //no update info received, what's wrong? Check license info, maybe expired or disabled
			$this->license_info_http_request($httpData); //refresh the license info
		}

		return $response;
		
	}


	public function get_plugin_update_info(){

		$pluginInfo = get_transient( 'xoo-plugin-info-'.$this->slug );

		if( $this->is_license_active() ){
			if( $pluginInfo === false ){ //transient expired, fetch again
				$pluginInfo = $this->plugin_update_info_http_request();
			}
		}
		else{
			$pluginInfo = false;
		}
		
		set_transient( 'xoo-plugin-info-'.$this->slug, $pluginInfo, DAY_IN_SECONDS );

		return $pluginInfo;

	}

	public function get_db_license_data(){

		$licenseInfo = $this->get_license_info();

		if( !$licenseInfo ){
			$licenseInfo = array();
		}

		$licenseInfo['plugin_slug'] = $this->slug;
		$licenseInfo['site_url'] 	= wp_parse_url(get_site_url())['host'];

		return $licenseInfo;
	}

	public function check_for_plugin_update( $transient ){

		
		if ( empty($transient->checked ) ) return $transient;

		$info 			= $this->get_plugin_update_info();

		$pluginVersion 	= get_plugin_data( ($this->helperArgs)['mainfile'] )['Version'];
	

		if( $info && version_compare( $pluginVersion, $info['version'], '<' ) && version_compare( $info['requires'], get_bloginfo( 'version' ), '<=' ) && version_compare( $info['requires_php'], PHP_VERSION, '<' ) ) {
			$result 				= new stdClass();
			$result->slug 			= $this->slug;
			$result->plugin 		= plugin_basename( ($this->helperArgs)['mainfile'] ) ;
			$result->new_version 	= $info['version'];
			$result->tested 		= $info['tested'];
			$result->package 		= $info['download_url'];

			$transient->response[ $result->plugin ] = $result;

		}

		return $transient;
	}


	public function show_plugin_info( $result, $action, $args  ){

		if ( $action !== 'plugin_information' || $args->slug !== $this->slug  )  {
			return $result;
		}
	
		// get updates
		$info = $this->get_plugin_update_info();

		if( !$info ) {
			return $result;
		}

		$result = new stdClass();

		$result->name 				= $info['name'];
		$result->slug 				= $info['slug'];
		$result->version 			= $info['version'];
		$result->tested 			= $info['tested'];
		$result->requires 			= $info['requires'];
		$result->author 			= $info['author'];
		$result->author_profile 	= $info['author_profile'];
		$result->download_link 		= $info['download_url'];
		$result->trunk 				= $info['download_url'];
		$result->requires_php 		= $info['requires_php'];
		$result->last_updated 		= $info['last_updated'];
		$result->sections 			= array(
			'description' => $info['sections']['description'],
			'installation' => $info['sections']['installation'],
			'changelog' => $info['sections']['changelog']
		);

		if( ! empty( $info['banners'] ) ) {
			$result->banners = array(
				'low' => $info['banners']['low'],
				'high' => $info['banners']['high']
			);
		}

		return $result;
	}

}



?>
