<?php

/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       http://designingmedia.com
 * @since      1.0.0
 *
 * @package    Wp_Whmcs_Sync
 * @subpackage Wp_Whmcs_Sync/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    Wp_Whmcs_Sync
 * @subpackage Wp_Whmcs_Sync/includes
 * @author     Designingmedia <http://designingmedia.com/>
 */
class Wp_Whmcs_Sync {

	/**
	 * The loader that's responsible for maintaining and registering all hooks that power
	 * the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      Wp_Whmcs_Sync_Loader    $loader    Maintains and registers all hooks for the plugin.
	 */
	protected $loader;

	/**
	 * The unique identifier of this plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $plugin_name    The string used to uniquely identify this plugin.
	 */
	protected $plugin_name;

	/**
	 * The current version of the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $version    The current version of the plugin.
	 */
	protected $version;

	/**
	 * Define the core functionality of the plugin.
	 *
	 * Set the plugin name and the plugin version that can be used throughout the plugin.
	 * Load the dependencies, define the locale, and set the hooks for the admin area and
	 * the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function __construct() {
		if ( defined( 'WP_WHMCS_SYNC_VERSION' ) ) {
			$this->version = WP_WHMCS_SYNC_VERSION;
		} else {
			$this->version = '1.0.0';
		}
		$this->plugin_name = 'wp-whmcs-sync';

		$this->load_dependencies();
		$this->set_locale();
		$this->define_admin_hooks();
		$this->define_public_hooks();

	}

	/**
	 * Load the required dependencies for this plugin.
	 *
	 * Include the following files that make up the plugin:
	 *
	 * - Wp_Whmcs_Sync_Loader. Orchestrates the hooks of the plugin.
	 * - Wp_Whmcs_Sync_i18n. Defines internationalization functionality.
	 * - Wp_Whmcs_Sync_Admin. Defines all hooks for the admin area.
	 * - Wp_Whmcs_Sync_Public. Defines all hooks for the public side of the site.
	 *
	 * Create an instance of the loader which will be used to register the hooks
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function load_dependencies() {

		/**
		 * The class responsible for orchestrating the actions and filters of the
		 * core plugin.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-wp-whmcs-sync-loader.php';

		/**
		 * The class responsible for defining internationalization functionality
		 * of the plugin.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-wp-whmcs-sync-i18n.php';

		/**
		 * The class responsible for defining all actions that occur in the admin area.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/class-wp-whmcs-sync-admin.php';

		/**
		 * The class responsible for defining all whmcs shortcode callbacks
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'public/class-wp-whmcs-sync-shortcode-handler.php';

		/**
		 * The class responsible for parsing content from the WHMCS url
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'public/class-wp-whmcs-sync-parser.php';


		/**
		 * The class responsible for defining all actions that occur in the public-facing
		 * side of the site.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'public/class-wp-whmcs-sync-public.php';
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'public/class-whmcs-http-sync-handler-public.php';
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'public/class-wp-whmcs-sync-simple-html-dom-public.php';

		/**
		 * Class responsible for creating meta fields
		 */
		if ( class_exists( 'CSF' ) ) {

			require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-wp-whmcs-sync-meta.php';
			require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/meta-boxes/class-wp-whmcs-sync-admin-meta.php';
			
			new Wp_Whmcs_Sync_Admin_Admin_Meta( $this->plugin_name );
		}

		$this->loader = new Wp_Whmcs_Sync_Loader();

	}

	/**
	 * Define the locale for this plugin for internationalization.
	 *
	 * Uses the Wp_Whmcs_Sync_i18n class in order to set the domain and to register the hook
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function set_locale() {

		$plugin_i18n = new Wp_Whmcs_Sync_i18n();

		$this->loader->add_action( 'plugins_loaded', $plugin_i18n, 'load_plugin_textdomain' );

	}

	/**
	 * Register all of the hooks related to the admin area functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_admin_hooks() {

		$plugin_admin = new Wp_Whmcs_Sync_Admin( $this->get_plugin_name(), $this->get_version() );

		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_styles' );
		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts' );

		//hook to validate data and creation of WHMCS-sync page
		$this->loader->add_action( "csf_wp-whmcs-sync-admin-page_save_before", $plugin_admin, 'whmcs_csf_before_save_callback' );
        $this->loader->add_action( "admin_enqueue_scripts", $plugin_admin, 'whmcs_enqueue_custom_admin_css' );

		//AJAX request for codestar get-in-touch form
		$this->loader->add_action( 'wp_ajax_get_in_touch', $plugin_admin, 'get_in_touch_handler' );
		$this->loader->add_action( 'wp_ajax_nopriv_get_in_touch', $plugin_admin, 'get_in_touch_handler' );

		//AJAX request to validate envato purchase code
		$this->loader->add_action( 'wp_ajax_whmcs_envato_validation', $plugin_admin, 'whmcs_envato_validation_handler' );
		$this->loader->add_action( 'wp_ajax_nopriv_whmcs_envato_validation', $plugin_admin, 'whmcs_envato_validation_handler' );

		//AJAX request to clear cache
		$this->loader->add_action( 'wp_ajax_clear_cache', $plugin_admin, 'clear_cache_handler' );
		$this->loader->add_action( 'wp_ajax_nopriv_clear_cache', $plugin_admin, 'clear_cache_handler' );

		$this->loader->add_action( 'init', $plugin_admin, 'custom_whmcs_rewrite_rules' );

		//whmcs sync service reminders
		$this->loader->add_action( 'whmcs_sync_service_reminders', $plugin_admin, 'whmcs_sync_service_reminders' );
	}

	/**
	 * Register all of the hooks related to the public-facing functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_public_hooks() {

		$plugin_public = new Wp_Whmcs_Sync_Public( $this->get_plugin_name(), $this->get_version() );

		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_styles' );
		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_scripts' );
		$this->loader->add_filter( 'wp_footer', $plugin_public, 'enqueue_footer_scripts' );

		//hook to render content on the sync page
		$this->loader->add_filter( 'the_content', $plugin_public, 'load_whmcs_sync_content', 10, 2 );

		//hook to trigger head events
		$this->loader->add_action( 'wp_head', $plugin_public, 'load_whmcs_sync_head', 10 );

		$this->loader->add_action( 'template_redirect', $plugin_public, 'remove_wpseo_for_whmcs_sync_page' );

		//AJAX request to check login status for WHMCS User
		$this->loader->add_action( 'wp_ajax_whmcs_login_status', $plugin_public, 'whmcs_login_status' );
		$this->loader->add_action( 'wp_ajax_nopriv_whmcs_login_status', $plugin_public, 'whmcs_login_status' );

		//AJAX request to check the avaialble whmcs domain
		$this->loader->add_action( 'wp_ajax_whmcs_ajax_domain_search', $plugin_public, 'whmcs_ajax_domain_search' );
		$this->loader->add_action( 'wp_ajax_nopriv_whmcs_ajax_domain_search', $plugin_public, 'whmcs_ajax_domain_search' );

		//AJAX request to validate whmcs login details
		$this->loader->add_action( 'wp_ajax_whmcs_validate_login', $plugin_public, 'whmcs_validate_login' );
		$this->loader->add_action( 'wp_ajax_nopriv_whmcs_validate_login', $plugin_public, 'whmcs_validate_login' );

		//AJAX request open whmcs ticket
		$this->loader->add_action( 'wp_ajax_whmcs_open_ticket', $plugin_public, 'whmcs_open_ticket' );
		$this->loader->add_action( 'wp_ajax_nopriv_whmcs_open_ticket', $plugin_public, 'whmcs_open_ticket' );

		//whmcs product billing cycles
		$this->loader->add_action( 'wp_ajax_whmcs_product_billing_cycle', $plugin_public, 'whmcs_product_billing_cycle_handler' );
		$this->loader->add_action( 'wp_ajax_nopriv_whmcs_product_billing_cycle', $plugin_public, 'whmcs_product_billing_cycle_handler' );

		//whmcs clientarea ajax
		$this->loader->add_action( 'wp_ajax_whmcs_clientarea', $plugin_public, 'whmcs_clientarea_handler' );
		$this->loader->add_action( 'wp_ajax_nopriv_whmcs_clientarea', $plugin_public, 'whmcs_clientarea_handler' );

		//hook to remove theme's style 
		$this->loader->add_action( 'wp_print_styles', $plugin_public, 'remove_hostiko_theme_style', 999 );

		//hook to trigger wp_loaded events
		$this->loader->add_action( 'wp_loaded', $plugin_public, 'whmcs_changes_from_url_param' );

		//whmcs invoices ajax
		$this->loader->add_action( 'wp_ajax_whmcs_invoices', $plugin_public, 'whmcs_invoices_handler' );
		$this->loader->add_action( 'wp_ajax_nopriv_whmcs_invoices', $plugin_public, 'whmcs_invoices_handler' );

		//whmcs credit balance ajax
		$this->loader->add_action( 'wp_ajax_whmcs_credit_balance', $plugin_public, 'whmcs_credit_balance_handler' );
		$this->loader->add_action( 'wp_ajax_nopriv_whmcs_credit_balance', $plugin_public, 'whmcs_credit_balance_handler' );

		//whmcs delete quote
		$this->loader->add_action( 'wp_ajax_whmcs_delete_quote', $plugin_public, 'whmcs_delete_quote_handler' );
		$this->loader->add_action( 'wp_ajax_nopriv_whmcs_delete_quote', $plugin_public, 'whmcs_delete_quote_handler' );

	}

	/**
	 * Run the loader to execute all of the hooks with WordPress.
	 *
	 * @since    1.0.0
	 */
	public function run() {
		$this->loader->run();
	}

	/**
	 * The name of the plugin used to uniquely identify it within the context of
	 * WordPress and to define internationalization functionality.
	 *
	 * @since     1.0.0
	 * @return    string    The name of the plugin.
	 */
	public function get_plugin_name() {
		return $this->plugin_name;
	}

	/**
	 * The reference to the class that orchestrates the hooks with the plugin.
	 *
	 * @since     1.0.0
	 * @return    Wp_Whmcs_Sync_Loader    Orchestrates the hooks of the plugin.
	 */
	public function get_loader() {
		return $this->loader;
	}

	/**
	 * Retrieve the version number of the plugin.
	 *
	 * @since     1.0.0
	 * @return    string    The version number of the plugin.
	 */
	public function get_version() {
		return $this->version;
	}

}