<?php

/**
 * The public-facing functionality of the plugin.
 *
 * @link       http://designingmedia.com
 * @since      1.0.0
 *
 * @package    Wp_Whmcs_Sync
 * @subpackage Wp_Whmcs_Sync/public
 */

/**
 * The public-facing functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the public-facing stylesheet and JavaScript.
 *
 * @package    Wp_Whmcs_Sync
 * @subpackage Wp_Whmcs_Sync/public
 * @author     Designingmedia <http://designingmedia.com>
 */
// define( 'HOUR_IN_SECONDS', 3600 );
class Wp_Whmcs_Sync_Public {

	use Whmcs_Sync_Shortcode_Handler;
	use Whmcs_Sync_Parser;

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	// private $meta_data;
	private $whmcs_page_title;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param      string    $plugin_name       The name of the plugin.
	 * @param      string    $version    The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {

		$this->plugin_name = $plugin_name;
		$this->version = $version;
		$this->load_shortcodes();

		//all whmcs sync settings
		$meta_options = get_option( 'wp-whmcs-sync-admin-page', true );
		if(is_array($meta_options)) {
			$this->meta_data = $meta_options;
		}

		if(!empty(get_option( '_whmcs_sync_page_id' ))){
			$whmcs_page = get_post(get_option( '_whmcs_sync_page_id' ));
			if($whmcs_page){
				$this->whmcs_page_title = $whmcs_page->post_title;
			}
		} else {
			$this->whmcs_page_title = 'WHMCS-Sync';
		}

	}

	/**
	 * Register the stylesheets for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Wp_Whmcs_Sync_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Wp_Whmcs_Sync_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_style( 'whmcs-custom-css', plugin_dir_url( __FILE__ ) . 'css/wp-whmcs-sync-custom.css', array(), $this->version, 'all' );
		wp_enqueue_style( 'datatable-css', 'https://cdn.datatables.net/1.10.24/css/jquery.dataTables.css', array(), $this->version, 'all' );
		wp_enqueue_style( 'selectize', 'https://cdnjs.cloudflare.com/ajax/libs/selectize.js/0.13.3/css/selectize.min.css', array(), $this->version, 'all' );

		if ( is_page( $this->whmcs_page_title ) ){
			wp_enqueue_style( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'css/wp-whmcs-sync-public.css', array(), $this->version, 'all' );
		}
	}

	/**
	 * Register the JavaScript for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Wp_Whmcs_Sync_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Wp_Whmcs_Sync_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_script( 'whmcs-shortcodes-js', plugin_dir_url( __FILE__ ) . 'js/wp-whmcs-sync-shortcodes.js', array( 'jquery' ), $this->version, false );
		wp_enqueue_script( 'datatable-js', 'https://cdn.datatables.net/1.10.24/js/jquery.dataTables.js', array( 'jquery' ), $this->version, false );
		wp_enqueue_script( 'sweeetalert', 'https://cdn.jsdelivr.net/npm/sweetalert2@11.10.0/dist/sweetalert2.all.min.js', array( 'jquery' ), $this->version, false );
		wp_enqueue_script( 'selectize', 'https://cdnjs.cloudflare.com/ajax/libs/selectize.js/0.13.3/js/standalone/selectize.min.js', array( 'jquery' ), $this->version, false );

		// if(is_page($this->whmcs_page_title)){
			wp_enqueue_script( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'js/wp-whmcs-sync-public.js', array( 'jquery' ), $this->version, false );
			wp_enqueue_script( 'popper-js', 'https://cdnjs.cloudflare.com/ajax/libs/popper.js/2.9.3/umd/popper.min.js', array( 'jquery' ), $this->version, false );

			if(is_array($this->meta_data)){
				if(is_array($this->meta_data['whmcs-color-group'])){
					$whmcs_page = '';
					if(is_page($this->whmcs_page_title) == true){
						$whmcs_page = 'true';
					}

					$created_whmcs_page = get_post(get_option( '_whmcs_sync_page_id' ));
					$localization_data = array(
						'whmcs_sync_style'     => $this->meta_data['whmcs-sync-style'],
						'whmcs_sync_template'  => $this->meta_data['whmcs-template'],
						'whmcs_loader_css'     => $this->meta_data['whmcs-loader-css'],
						'whmcs_loader_enabled' => $this->meta_data['whmcs-enable-loader-css'],
						'whmcs_url'  		   => $this->meta_data['whmcs-url'],
						'whmcs_hostiko_layout' => $this->meta_data['whmcs-custom-layout'],
						'whmcs_sso' 		   => isset($this->meta_data['whmcs-sync-sso']) ? $this->meta_data['whmcs-sync-sso'] : '',
						'whmcs_api' 		   => isset($this->meta_data['whmcs-api']) ? $this->meta_data['whmcs-api'] : '',
						'whmcs_page' 		   => $whmcs_page,
						'created_whmcs_page'   => $created_whmcs_page ? 'true' : 'false',
						'whmcs_disable_header' => $this->meta_data['whmcs-disable-hostiko-header-footer'],
						'font_family'          => $this->meta_data['whmcs-sync-font-family'],
						'friendly_url'		   => isset($this->meta_data['whmcs-friendly-url-support']) ? $this->meta_data['whmcs-friendly-url-support'] : '',
						'whmcs_disable_theme_header' => isset($this->meta_data['whmcs-disable-theme-header-footer']) ? $this->meta_data['whmcs-disable-theme-header-footer'] : '',
					);

					wp_localize_script($this->plugin_name, 'whmcs_data', $localization_data);
				}
			}
			$urls = [
				'siteUrl' => site_url( ),
				'pluginDir' => plugin_dir_url( __FILE__ ),
				'ajaxUrl' => admin_url('admin-ajax.php'),
				'whmcsSync' => get_option( '_whmcs_sync_page_id' ) ? get_post(get_option( '_whmcs_sync_page_id' ))->guid : '',
				'whmcsSlug' => get_option( '_whmcs_sync_page_id' ) ? get_post_field('post_name', get_option( '_whmcs_sync_page_id' )) : '',
				'whmcsSyncPageData' => get_option( '_whmcs_sync_page_id' ) ? get_post(get_option( '_whmcs_sync_page_id' )) : '',
			];

			wp_localize_script( $this->plugin_name, 'urls_to_frontend', $urls );
		// }

	}

	/**
	 * Method to load templates and layouts directly from the url
	 */
	public function whmcs_changes_from_url_param(){
		if(is_page($this->whmcs_page_title)){
			$update = false;
			//change template
			if((isset($_GET['temp']) && $_GET['temp'] === 'hostiko') && (isset($_GET['layout']) && !empty($_GET['layout']))){
				if(!empty($_GET['layout'])){
					$this->meta_data['whmcs-custom-layout'] = $_GET['layout'];
					$update = true;
				}
			}

			if(isset($_GET['temp']) && !empty($_GET['temp'])){
				$this->meta_data['whmcs-template'] = $_GET['temp'];
				$update = true;
			}

			//apply cucstom css
			if(isset($_GET['style']) && $_GET['style'] === 'yes'){
				$this->meta_data['whmcs-sync-style'] = true;
				$update = true;
			} else if(isset($_GET['style']) && $_GET['style'] === 'no') {
				$this->meta_data['whmcs-sync-style'] = false;
				$update = true;
			}

			//enable or disable footer
			if(isset($_GET['hf']) && $_GET['hf'] === 'yes'){
				$this->meta_data['whmcs-disable-theme-header-footer'] = true;
			} else if(isset($_GET['hf']) && $_GET['hf'] === 'no') {
				$this->meta_data['whmcs-disable-theme-header-footer'] = false;
			}

			if($update == true){
				update_option('wp-whmcs-sync-admin-page', $this->meta_data);
			}

			if(isset($_POST['whmcs-api-url']) && isset($_POST['whmcs-api-identifier']) && isset($_POST['whmcs-api-secret'])){
				if(!empty($_POST['whmcs-api-url']) && !empty($_POST['whmcs-api-identifier']) && !empty($_POST['whmcs-api-secret'])){
					update_option('_api_key_management', $_POST);
				}
			}
		}
	}

	/**
	 * Enqueue footer scripts and styles.
	 * This method is responsible for loading necessary scripts and styles in the footer section.
	 */
	public function enqueue_footer_scripts() {
		if ( is_page( $this->whmcs_page_title ) ) {
			// Enqueue Font Awesome styles from a CDN.
			wp_enqueue_style('font-awesome-style', 'https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.2/css/all.min.css', array(), $this->version, 'all');

			// Check if the 'whmcs-footer' meta data is set to 'site'.
			if ( $this->meta_data['whmcs-footer'] === 'site' ) {
				// Call the method to display the WHMCS Sync footer.
				$this->wp_whmcs_sync_footer();
			}
		}
	}


	// public function t1ZEx($U1DdE) { echo "\x3c\160\162\x65\x3e"; print_r($U1DdE); }
	public function pre($data){
		echo '<pre>'; print_r($data);
	}

	/**
	 * Display the footer message for WHMCS Sync.
	 * This method generates and displays a centered footer message.
	 *
	 * @param bool $nodisplay Set to true to return the message without displaying, false to echo the message.
	 * @return string|null If $nodisplay is true, returns the generated footer message; otherwise, echoes the message.
	 */
	public function wp_whmcs_sync_footer( $display_footer = '' ) {
		$footer = false;
		// Check if in the WordPress admin or on a WordPress admin page
		$is_admin_request = (defined('WP_ADMIN') && WP_ADMIN == true) || (strpos($_SERVER['PHP_SELF'], 'wp-admin') !== false);

		// If in the admin, return early
		if ($is_admin_request) {
			return $footer;
		}

		// Generate the footer message
		$footer_msg = '<center style="margin-top:0px;font-size:small">';
			$footer_msg .= 'WordPress and WHMCS integration by <a target="_blank" href="https://designingmedia.com/" target="_blank">Designingmedia</a>';
		$footer_msg .= '</center>';

		// Set a flag to indicate the footer has been generated
		$wp_whmcs_sync_footer_generated = true;

		// Return or echo the message based on the $display_footer parameter
		if ($display_footer === true) {
			return $footer_msg;
		} else {
			echo $footer_msg;
		}
	}

	/**
	 * Method to add the shortcode
	 *
	 * @return void
	 */
	public function load_shortcodes() {	
		add_shortcode( 'whmcs-login-url', array( $this, 'whmcs_login_url_shortcode_callback' ) );
		add_shortcode( 'whmcs-cart-redirect', array( $this, 'whmcs_cart_redirect_shortcode_callback' ) );
		add_shortcode( 'whmcs-domain-register', array( $this, 'whmcs_domain_register_shortcode_callback' ) );
		add_shortcode( 'whmcs-domain-transfer', array( $this, 'whmcs_domain_transfer_shortcode_callback' ) );
		add_shortcode( 'whmcs-product-url', array( $this, 'whmcs_product_url_shortcode_callback' ) );
		add_shortcode( 'whmcs-ajax-domain-searh', array( $this, 'whmcs_ajax_domain_search_shortcode_callback' ) );
		add_shortcode( 'whmcs-service-listing', array( $this, 'whmcs_service_listing_shortcode_callback' ) );
		add_shortcode( 'open-ticket', array( $this, 'whmcs_support_ticket_shortcode_callback' ) );
		add_shortcode( 'get-tickets', array( $this, 'whmcs_get_tickets_callback' ) );
		add_shortcode( 'whmcs-clientarea', array( $this, 'whmcs_clientarea_shortcode_callback' ) );
		add_shortcode( 'whmcs-invoice-listing', array( $this, 'whmcs_invoices_shortcode_callback' ) );
		add_shortcode( 'whmcs-announcements', array( $this, 'whmcs_accouncements_shortcode_callback' ) );
		add_shortcode( 'whmcs-affiliate', array( $this, 'whmcs_affiliate_shortcode_callback' ) );
		add_shortcode( 'whmcs-credit-balance', array( $this, 'whmcs_credit_balance_shortcode_callback' ) );
		add_shortcode( 'whmcs-dynamic-pricing-table', array( $this, 'whmcs_dynamic_pricing_table_shortcode_callback' ) );
		add_shortcode( 'whmcs-api-management', array( $this, 'whmcs_api_management_shortcode_callback' ) );
		add_shortcode( 'whmcs-multi-language', array( $this, 'whmcs_multi_language_shortcode_callback') );
		add_shortcode( 'whmcs-usage-statistics', array( $this, 'whmcs_usage_statistics_shortcode_callback') );
		add_shortcode( 'whmcs-service-renewal-reminders', array( $this, 'whmcs_service_renewal_shortcode_callback') );
		add_shortcode( 'whmcs-custom-dashboard-widgets', array( $this, 'whmcs_custom_dashboard_widgets_shortcode_callback') );
		add_shortcode( 'whmcs-get-payment-methods', array( $this, 'whmcs_get_payment_methods_shortcode_callback') );
		add_shortcode( 'whmcs-get-quotes', array( $this, 'whmcs_get_quotes_shortcode_callback') );
	}

	/**
	 * Get the ID of the main page for WHMCS Sync.
	 * This method retrieves the post ID of the WHMCS Sync page.
	 *
	 * @return int The post ID of our whmcs sync page.
	 */
	public function wp_whmcs_sync_main_page() {
		global $post;
		// Retrieve and return the post ID of the WHMCS Sync page
		if($post->post_title === $this->whmcs_page_title){
			return $post->ID;
		}
	}

	/**
	 * Method to fix some redirection urls
	 */
	private function fix_whmcs_urls($post){
		if(!empty(get_option( '_whmcs_sync_page_id' ))){
			if(get_post_field('post_name', get_option( '_whmcs_sync_page_id' )) === $post->post_name){
				// Get the current URL
				$current_url = "http" . (isset($_SERVER['HTTPS']) ? "s" : "") . "://$_SERVER[HTTP_HOST]$_SERVER[REQUEST_URI]";

				// Define an array of patterns to match and the corresponding query strings
				$patterns_and_queries = array(
					'/\/announcements\//' => '/?ccce=announcements/',
					'/\/knowledgebase\//' => '/?ccce=knowledgebase/',
					'/\/download\//' => '/?ccce=download/',
					// Add more patterns and queries as needed
				);

				// Check if any pattern matches the URL
				foreach ($patterns_and_queries as $pattern => $query_string) {
					if (strpos($current_url, $query_string) === false && preg_match($pattern, $current_url)) {
						// Add the query string and trailing slash before the matched pattern in the URL
						$current_url = preg_replace($pattern, $query_string, $current_url, 1);
						echo '<script type="text/javascript">window.location.href = "' . $current_url . '";</script>';
						exit();
					}
				}

				$redirects = array(
					'/\/announcements$/' => '/?ccce=announcements',
					'/\/knowledgebase$/' => '/?ccce=knowledgebase',
					'/\/download$/' => '/?ccce=download',
					// Add more patterns and redirects as needed
				);

				// Check if any pattern matches the URL
				foreach ($redirects as $pattern => $query_string) {
					if (preg_match($pattern, $current_url)) {
						// Redirect to the new URL
						$new_url = preg_replace($pattern, $query_string, $current_url);
						echo '<script type="text/javascript">window.location.href = "' . $new_url . '";</script>';
						exit();
					}
				}
			}

			$current_url = "http://$_SERVER[HTTP_HOST]$_SERVER[REQUEST_URI]";

			// Define an array of search strings and their corresponding replacements
			$search_settings = [
				'?ccce=https:akdesigner.comcart' => '?ccce=cart',
				'?ccce=https:akdesigner.comwhmcs-templatescart' => '?ccce=cart',
			];

			// Check if any of the search strings are present in the current URL
			foreach ($search_settings as $search => $replace) {
				if (strpos($current_url, $search) !== false) {
					// Update the relevant part of the URL
					$updatedURL = str_replace($search, $replace, $current_url);

					// Redirect to the updated URL
					header("Location: $updatedURL");
					exit; // Make sure to exit after sending the header
				}
			}
		}
	}

	// Function to minify CSS content
	private function minify_css($css) {
		// Remove comments
		$css = preg_replace('!/\*[^*]*\*+([^/][^*]*\*+)*/!', '', $css);
		// Remove tabs, spaces, newlines, etc.
		$css = str_replace(array("\r\n", "\r", "\n", "\t", '  ', '    ', '    '), '', $css);
		return $css;
	}

	/**
	 * Method to remove SEO feature for the sync page
	 */
	public function remove_wpseo_for_whmcs_sync_page(){
		if(get_option( '_whmcs_sync_page_id' )){
			if ( is_page ( get_option( '_whmcs_sync_page_id' ) ) ) {
				if (defined('WPSEO_VERSION') && class_exists('Yoast\WP\SEO\Integrations\Front_End_Integration')) {
					$front_end = YoastSEO()->classes->get( Yoast\WP\SEO\Integrations\Front_End_Integration::class );
		
					remove_action( 'wpseo_head', [ $front_end, 'present_head' ], -9999 );
				}
			}
		}
	}

	/**
	 * Load WHMCS Sync Head.
	 * This function is responsible for loading necessary scripts and styles for WHMCS-Sync.
	 */
	public function load_whmcs_sync_head() {
		if(is_page($this->whmcs_page_title)){
			ob_start();
			global $post;
	
			// $this->fix_whmcs_urls($post);
			// $this->t1ZEx('asdfasdf'); exit;
	
			// Check if $post is set and has an ID
			if (!(isset($post->ID))) {
				return;
			}
	
			// Check if 'ccce' is set in the request or if post title is 'WHMCS-Sync'
			if (isset($_REQUEST['ccce']) || (isset($post->post_title) && $post->post_title === $this->whmcs_page_title)) {
	
				// Check if AJAX is set and in the allowed values (1 or 2)
				if (isset($_REQUEST['ajax']) && in_array($_REQUEST['ajax'], array(1, 2))) {
					return;
				}
	
				//if cache feature is enabled
				if(isset($this->meta_data['whmcs-sync-cache']) && $this->meta_data['whmcs-sync-cache'] == true){
					// Path to your cache folder
					$cache_folder = plugin_dir_path(__FILE__) . 'cache/';

					$files = glob($cache_folder . '*');

					// Check if there are files in the cache directory
					if (!empty($files)) {
						// Enqueue each file based on its extension
						foreach ($files as $file) {
							$file_url = plugin_dir_url(__FILE__) . 'cache/' . basename($file);
							$file_extension = pathinfo($file, PATHINFO_EXTENSION);

							if ($file_extension === 'css') {
								// Enqueue CSS file
								wp_enqueue_style('cached-style', $file_url);
							} elseif ($file_extension === 'js') {
								// Enqueue JavaScript file
								// wp_enqueue_script('cached-script', $file_url, array(), false);
								echo '<script src="'.$file_url.'"></script>';
							} else {
								echo '<script type="text/javascript"> '.file_get_contents($file).' </script>';
							}
						}
					} else {
						$wp_whmcs_sync_head = $this->parse_whmcs_sync_content();
						if ( isset( $wp_whmcs_sync_head['head'] ) ) {
							// Parse the HTML content to extract the URLs of specific files
							$dom = new DOMDocument();
							$dom->loadHTML( $wp_whmcs_sync_head['head'] );
							
							$links = $dom->getElementsByTagName( 'link' );
							$scripts = $dom->getElementsByTagName( 'script' );
							$css_content = '';
							$files_to_cache = array();

							// Extract URLs of specific CSS files to cache
							foreach ($links as $link) {
								if ($link->getAttribute('rel') == 'stylesheet') {
									$href = $link->getAttribute('href');
									// Check if the URL matches the ones you want to cache
									if (strpos($href, site_url(get_post_field('post_name', get_option('_whmcs_sync_page_id')).'/')) !== false) {
										$css_content .= file_get_contents($href);
									}
								}
							}

							// Minify CSS content
							$minified_css = $this->minify_css($css_content);

							// Cache the minified CSS content into a single file
							$cache_file = $cache_folder . 'cached.min.css';
							file_put_contents($cache_file, $minified_css);
							
							foreach ( $scripts as $script ) {
								$crf_token = $script->nodeValue;
								$src = $script->getAttribute( 'src' );
								// Check if the URL matches the ones you want to cache
								if ( strpos( $src, site_url(get_post_field('post_name', get_option( '_whmcs_sync_page_id' )).'/') ) !== false ) {
									$files_to_cache[] = $src;
								}
							}


							// Cache the files individually
							foreach ( $files_to_cache as $file ) {
								$file_content = file_get_contents( $file );
								// Generate cache file name based on the file URL
								$cache_file_individual = $cache_folder . basename( $file );

								file_put_contents( isset(explode('?', $cache_file_individual)[0]) ? explode('?', $cache_file_individual)[0] : $cache_file_individual, $file_content );

							}
							
							// Enqueue the combined and minified CSS file
							// wp_enqueue_style('custom-style', plugin_dir_url(__FILE__) . 'cache/combined.min.css');
							foreach ( $scripts as $key => $script ) {
								if($key == 0){
									// Cache the content of the head
									file_put_contents( $cache_folder . 'whmcs_sync_head_cache', $script->nodeValue );
								}
							}
							
							echo $wp_whmcs_sync_head['head'];
						}
					}
				} else {
					$wp_whmcs_sync_head = $this->parse_whmcs_sync_content();
					if (isset($wp_whmcs_sync_head['head'])) {
						echo $wp_whmcs_sync_head['head'];
					}
				}
				
				//load WHMCS-Sync own css
				if(is_array($this->meta_data)){
					if($this->meta_data['whmcs-sync-style'] == true && $this->meta_data['whmcs-template'] != 'hostiko'){
						wp_enqueue_style('whmcs-sync-style', plugin_dir_url(__FILE__) . 'css/wp-whmcs-sync-default.css', array(), $this->version, 'all', 'screen', 10);
					}
				}
	
				// Output custom CSS from options
				if(is_array($this->meta_data)){
					if(!empty($this->meta_data['whmcs-custom-css'])){
						echo '<style type="text/css">' . $this->meta_data['whmcs-custom-css'] . '</style>';
					}
	
					// Output jQuery script if 'whmcs-jquery-library' is set to 'wp'
					if($this->meta_data['whmcs-jquery-library'] == 'wp'){
						echo '<script type="text/javascript">$=jQuery;</script>';
					}
				}
			}
		}
	}

	/**
	 * Method to remove hostiko theme style for whmcs-sync page
	 */
	public function remove_hostiko_theme_style(){
		if(is_page($this->whmcs_page_title) == true && $this->meta_data['whmcs-template'] === 'hostiko'){
			wp_dequeue_script('bootstrap-js');
			// wp_dequeue_style('header-style');
		}
	}

	/**
	 * Method to load content on the WHMCS-Sync page.
	 *
	 * This method is responsible for rendering content on the WHMCS-Sync page when it's the main query
	 * and the current page is a "WHMCS-Sync" page.
	 *
	 * @param string $original_content The current content of the page.
	 * @return string The modified content to include WHMCS-Sync content.
	 */
	public function load_whmcs_sync_content($original_content){
		$envato_validation = get_option( 'whmcs_sync_validate' );
		if($envato_validation == true){
			global $wp_whmcs_sync_content, $post;
			// Check if the current page is a "WHMCS-Sync" page
			if (is_page() && is_main_query()) {
				
				// Ensure the post title is "WHMCS-Sync"
				if ($post->post_title != $this->whmcs_page_title) {
					return $original_content;
				}
	
				// Parse WHMCS-Sync content
				$wp_whmcs_sync_content = $this->parse_whmcs_sync_content();

				if ($wp_whmcs_sync_content) {
					ob_start();
					$top_content = ob_get_clean();
					$main_content = '<div id="wp_whmcs_sync">';
						if (is_array($wp_whmcs_sync_content) && isset($wp_whmcs_sync_content['main'])) {
							// Add the main WHMCS-Sync content
							$main_content .= $wp_whmcs_sync_content['main'];
						}
					$main_content .= '</div>';
					ob_start();
					$bottom_content = ob_get_clean();
					// Combine top, main, and bottom content to create the modified content
					$modified_content = $top_content . $main_content . $bottom_content;
					return $modified_content;
				}
			}
	
			return $original_content;
		} else {
			return $original_content;
		}
		
	}

	/**
	 * Retrieves the WHMCS URL saved in the plugin settings.
	 *
	 * This method retrieves the WHMCS URL from the plugin's settings and ensures it's correctly formatted.
	 *
	 * @return string|false The WHMCS URL if found, or false if not available.
	 */
	public function wp_whmcs_sync_custom_url_handler() {
		if (!empty($this->meta_data['whmcs-url'])) {
			$whmcs_url = rtrim($this->meta_data['whmcs-url'], '/'); // Remove trailing slashes
			return $whmcs_url;
		}
		return false; // Return false if the URL is not available.
	}

	/**
	 * Builds the base URL and page ID for the current or specified WordPress page.
	 *
	 * This method constructs the base URL and page ID based on the current WordPress page's permalink or ID.
	 * It takes into account permalink structure settings and the option to force SSL.
	 *
	 * @param string &$base_url The resulting base URL.
	 * @param string &$page_id  The resulting page ID.
	 * @param bool   $is_current Indicates if the current page should be considered.
	 *
	 * @return string The generated base URL for the page.
	 */
	public function wp_whmcs_sync_main_screen(&$base_url, &$page_id, $is_current = false) {
		global $wordpress_page_name, $current_post;
		$current_page_id = isset($current_post) && $is_current ? $current_post->ID : get_the_ID();
		$page_permalink = get_permalink($current_page_id);
		$home_url = get_option('home');

		if ($is_current) {
			// Parse the permalink for the current page
			$matches = [];
			preg_match('/(.*)\?page_id\=(.*)/', $page_permalink, $matches);
			if (count($matches) == 3) {
				$page_id = '&page_id=' . $matches[2];
				$base_url = $matches[1];
			} else {
				$page_id = '';
				$base_url = $page_permalink;
			}
		} else {
			if (get_option('permalink_structure')) {
				// Handle permalink structure
				$wordpress_page_name = str_replace($home_url, '', $page_permalink);
				$wordpress_page_name = str_replace('https://', 'http://', $wordpress_page_name);
				$page_id = '';
				$base_url = $home_url . $wordpress_page_name;
				if (substr($base_url, -1) != '/') {
					$base_url .= '/';
				}
			} else {
				// Handle non-permalink structure
				$page_id = '&page_id=' . $current_page_id;
				$base_url = $home_url;
				if (substr($base_url, -1) != '/') {
					$base_url .= '/';
				}
			}
		}

		if ((isset($_SERVER['HTTPS']) && ($_SERVER['HTTPS'] == "on") || get_option('whmcs_sync_sso_force_ssl'))) {
			// If HTTPS is forced, update the URL to use HTTPS
			$secure_url = str_replace('http://', 'https://', $base_url);
			$secure_url = str_replace('http://', 'https://', $base_url);
		} else {
			// Use the original URL
			$secure_url = $base_url;
		}

		return $secure_url;
	}

	/**
	 * Logs messages for debugging and records them in the WordPress options.
	 *
	 * This method logs debugging messages, allowing developers to track the application's behavior.
	 * If debugging is enabled, it stores log entries in WordPress options for later analysis.
	 *
	 * @param string|int  $log_type   The type of log entry ('debug' by default).
	 * @param mixed       $log_msg    The message to log, which can be a string or an array.
	 * @param string      $file_name  The name of the file (optional, usually left empty).
	 * @param int         $line_num   The line number in the file (optional, usually left empty).
	 */
	public function whmcs_sync_logger($log_type = 0, $log_msg = '', $file_name = "", $line_num = 0) {
		// Set $log_type to 'debug' if it's 0
		if ($log_type == 0) {
			$log_type = 'debug';
		}
	
		// Check if debugging is enabled
		if ($this->meta_data['whmcs-debug'] == true) {

			// Convert $log_msg to JSON if it's an array
			if (is_array($log_msg)) {
				$log_msg = json_encode($log_msg);
			}
	
			// Get the current log entries
			$log_entries = get_option('whmcs_sync_logger_entries', []);
			if (!is_array($log_entries)) {
				$log_entries = []; // Initialize an empty array
			}
	
			// Add a new log entry at the beginning
			array_unshift($log_entries, [microtime(), $log_type, $log_msg]);
	
			// Limit the log to a maximum of 100 entries
			if (count($log_entries) > 100) {
				array_pop($log_entries);
			}

			// Update the log option with the modified entries
			update_option('whmcs_sync_logger_entries', $log_entries);
		}
	}

	/**
	 * Method to hit WHMCS api
	 */
	public function whmcs_api_calling($body){
		if($this->meta_data && !empty($this->meta_data['whmcs-api'])){
			$ch = curl_init();
			curl_setopt($ch, CURLOPT_URL, $this->meta_data['whmcs-api']);
			curl_setopt($ch, CURLOPT_POST, 1);
			curl_setopt($ch, CURLOPT_POSTFIELDS,
				http_build_query(
					$body
				)
			);
			curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
			$response = curl_exec($ch);
			curl_close($ch);
	
			return json_decode($response, true);
		}
	}

}
