<?php
namespace AIOSEO\Plugin\Extend\Image;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Adds support for Image SEO.
 *
 * @since 1.0.0
 */
class Image {
	/**
	 * Class constructor.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function __construct() {
		$this->hooks();
	}

	/**
	 * Registers our hooks.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	private function hooks() {
		// Filter images embedded into posts.
		add_filter( 'the_content', [ $this, 'filterContent' ] );
		// Filter images embedded in the short description of WooCommerce Products.
		add_filter( 'woocommerce_short_description', [ $this, 'filterContent' ] );
		// Filter attachment pages.
		add_filter( 'wp_get_attachment_image_attributes', [ $this, 'filterImageAttributes' ], 10, 2 );
	}

	/**
	 * Filters the content of the requested post.
	 *
	 * @since 1.0.0
	 *
	 * @param  string $content The post content.
	 * @return string $content The filtered post content.
	 */
	public function filterContent( $content ) {
		if ( is_admin() || ! is_singular() ) {
			return $content;
		}

		return preg_replace_callback( '/<img.*?>/', [ $this, 'filterEmbeddedImages' ], $content );
	}

	/**
	 * Filters the attributes of image attachment pages.
	 *
	 * @since 1.0.0
	 *
	 * @param  array   $attributes The image attributes.
	 * @param  WP_Post $post       The post object.
	 * @return array   $attributes The filtered image attributes
	 */
	public function filterImageAttributes( $attributes, $post ) {
		if ( is_admin() || ! is_singular() ) {
			return $attributes;
		}

		$attributes['title'] = $this->getAttribute( 'title', $post->ID );
		$attributes['alt']   = $this->getAttribute( 'altTag', $post->ID );

		return $attributes;
	}

	/**
	 * Filters the attributes of images that are embedded in the post content.
	 *
	 * Helper function for the filterContent() method.
	 *
	 * @since 1.0.0
	 *
	 * @param  array  $images The HTML image tag (first match of Regex pattern).
	 * @return string         The filtered HTML image tag.
	 */
	public function filterEmbeddedImages( $images ) {
		$image = $images[0];

		$id = $this->imageId( $image );
		if ( ! $id ) {
			return $images[0];
		}

		$title  = $this->findExistingAttribute( 'title', $image );
		$altTag = $this->findExistingAttribute( 'alt', $image );

		$image = $this->insertAttribute(
			$image,
			'title',
			$this->getAttribute( 'title', $id, $title )
		);

		$image = $this->insertAttribute(
			$image,
			'alt',
			$this->getAttribute( 'altTag', $id, $altTag )
		);

		return $image;
	}

	/**
	 * Tries to extract the attachment page ID of an image.
	 *
	 * @since 1.0.0
	 *
	 * @param  string $image The image HTML tag.
	 * @return mixed         The ID of the attachment page or false if no ID could be found.
	 */
	private function imageId( $image ) {
		// Check if class contains an ID.
		preg_match( '#wp-image-(\d+)#', $this->findExistingAttribute( 'class', $image ), $matches );

		if ( empty( $matches ) ) {
			return false;
		}

		return intval( $matches[1] );
	}

	/**
	 * Inserts a given value for a given image attribute.
	 *
	 * @since 1.0.0
	 *
	 * @param  string $image         The image HTML.
	 * @param  string $attributeName The attribute name.
	 * @param  string $value         The attribute value.
	 * @return array                 The modified image attributes.
	 */
	private function insertAttribute( $image, $attributeName, $value ) {
		if ( empty( $value ) ) {
			return $image;
		}

		$value = esc_attr( $value );

		$image = preg_replace( $this->attributeRegex( $attributeName, true, true ), '${1}' . $value . '${2}', $image, 1, $count );
		if ( ! $count ) {
			// Let's try single quotes.
			$image = preg_replace( $this->attributeRegex( $attributeName, false, true ), '${1}' . $value . '${2}', $image, 1, $count );
		}

		// Attribute does not exist. Let's append it at the beginning of the tag.
		if ( ! $count ) {
			$image = preg_replace( '/<img /', '<img ' . $this->attributeToHtml( $attributeName, $value ) . ' ', $image );
		}

		return $image;
	}

	/**
	 * Returns the value of a given image attribute.
	 *
	 * @since 1.0.0
	 *
	 * @param  string $attributeName The attribute name.
	 * @param  int    $id            The attachment page ID.
	 * @param  string $value         The value, if it already exists.
	 * @return string                The attribute value.
	 */
	private function getAttribute( $attributeName, $id, $value = '' ) {
		$format = aioseo()->options->image->format->$attributeName;

		if ( $value ) {
			// If the value already exists on the image (e.g. alt text on Image Block), use that to replace the relevant tag in the format.
			$tag    = 'title' === $attributeName ? '#image_title' : '#alt_tag';
			$format = aioseo()->helpers->pregReplace( "/$tag/", $value, $format );
		}

		$attribute = aioseoImageSeo()->tags->replaceTags( $format, $id, aioseo()->options->image->stripPunctuation->$attributeName );

		$snakeName = aioseo()->helpers->toSnakeCase( $attributeName );
		return apply_filters( "aioseo_image_seo_$snakeName", $attribute, [ $id ] );
	}

	/**
	 * Returns the value of the given attribute if it already exists.
	 *
	 * @since 1.0.6
	 *
	 * @param  string $attributeName The attribute name, "title" or "alt".
	 * @param  string $image         The image HTML.
	 * @return string                The value.
	 */
	private function findExistingAttribute( $attributeName, $image ) {
		$found = preg_match( $this->attributeRegex( $attributeName ), $image, $value );
		if ( ! $found ) {
			// Let's try single quotes.
			preg_match( $this->attributeRegex( $attributeName, false ), $image, $value );
		}

		return ! empty( $value ) ? $value[1] : false;
	}

	/**
	 * Returns a regex string to match an attribute.
	 *
	 * @since 1.0.7
	 *
	 * @param  string $attributeName      The attribute name.
	 * @param  bool   $useDoubleQuotes    Use double or single quotes.
	 * @param  bool   $groupReplaceValue  Regex groupings without the value.
	 * @return string                     The regex string.
	 */
	private function attributeRegex( $attributeName, $useDoubleQuotes = true, $groupReplaceValue = false ) {
		$quote = $useDoubleQuotes ? '"' : "'";

		$regex = $groupReplaceValue ? "/(\s%s=$quote).*?($quote)/" : "/\s%s=$quote(.*?)$quote/";

		return sprintf( $regex, trim( $attributeName ) );
	}

	/**
	 * Returns an attribute as HTML.
	 *
	 * @since 1.0.7
	 *
	 * @param  string $attributeName The attribute name.
	 * @param  string $value         The attribute value.
	 * @return string                The HTML formatted attribute.
	 */
	private function attributeToHtml( $attributeName, $value ) {
		return sprintf( '%s="%s"', $attributeName, esc_attr( $value ) );
	}
}