<?php

defined( 'ABSPATH' ) || exit;

class Play_User_Profile {

    protected static $_instance = null;
    private $users_can_register = null;
    private $page_login = null;
    private $email_activation = false;
    private $meta_key_activate_code = 'activation_code';
    private $meta_key_activate_link = 'activation_link';
    private $meta_key_activated = 'email_activated';
    private $form_wrapper = '<div id="login-form">%1$s</div>';

    public static function instance() {
        if ( is_null( self::$_instance ) ) {
            self::$_instance = new self();
        }

        return self::$_instance;
    }

    public function __construct() {
        $this->users_can_register = get_option( 'users_can_register' ) ? true : false;
        $this->email_activation   = play_get_option( 'email_activation' ) ? true : false;
        $this->page_login         = play_get_option( 'page_login' );

        add_action( 'template_redirect', array( $this, 'update_profile' ) );

        add_filter( 'login_form_middle', array( $this, 'login_form_middle' ), 10, 1 );
        add_filter( 'register_form_middle', array( $this, 'register_form_middle' ), 10, 1 );
        add_filter( 'lostpassword_form_middle', array( $this, 'lostpassword_form_middle' ), 10, 1 );

        add_filter( 'login_form_top', array( $this, 'login_form_top' ), 10, 1 );
        add_filter( 'login_form_bottom', array( $this, 'login_form_bottom' ), 10, 1 );
        add_filter( 'register_form_bottom', array( $this, 'register_form_bottom' ), 10, 1 );
        add_filter( 'lostpassword_form_bottom', array( $this, 'lostpassword_form_bottom' ), 10, 1 );
        add_action( 'admin_notices', array( $this, 'admin_notices' ), 1 );
        add_action( 'user_register', array( $this, 'user_register' ), 10, 1 );
        add_action( 'init', array( $this, 'verify_user_code' ), 9 );
        add_filter( 'wp_authenticate_user', array( $this, 'authenticate_user' ), 10, 2 );
        
        // login page
        add_filter( 'login_url', array( $this, 'login_url' ), 10, 3 );

        //add_filter( 'wp_mail', array( $this, 'wp_mail' ), 10 );
        add_filter( 'wp_new_user_notification_email', array( $this, 'new_user_notification_email' ), 10, 3 );

        add_filter( 'play_modal_login_form', array( $this, 'get_login_form' ) );

        add_shortcode( 'wp_login_form', array( $this, 'wp_login_form_shortcode' ) );
        add_shortcode( 'wp_register_form', array( $this, 'wp_register_form_shortcode' ) );
        add_shortcode( 'wp_lostpassword_form', array( $this, 'wp_lostpassword_form_shortcode' ) );
        add_shortcode( 'play_login_form', array( $this, 'login_form_shortcode' ) );

        add_action( 'template_redirect', array( $this, 'logged_in_redirect' ) );

        add_filter( 'manage_users_custom_column', array( $this, 'user_row' ), 10, 3 );
        add_filter( 'manage_users_columns', array( $this, 'user_column' ), 10, 1 );

        function wp_register_form( $args = array() ) {
            return Play_User_Profile::instance()->wp_register_form( $args );
        }

        function wp_lostpassword_form( $args = array() ) {
            return Play_User_Profile::instance()->wp_lostpassword_form( $args );
        }

        function wp_profile_form( $user_id ) {
            return Play_User_Profile::instance()->wp_profile_form( $user_id );
        }

        do_action( 'play_block_user_profile_init', $this );
    }

    public function wp_mail( $arr ) {
        $mail = get_option('_mail');
        update_option('_mail', $mail.json_encode($arr));
        return $arr;
    }

    public function new_user_notification_email($email, $user, $blogname){
        if( play_get_option( 'email_newuser' ) ){
            $email['subject'] = $this->replace_token( play_get_option( 'email_newuser_subject' ), $user );
            $email['message'] = $this->replace_token( play_get_option( 'email_newuser_content' ), $user );
        }
        return apply_filters('play_block_newuser_email', $email);
    }

    public function replace_token($str, $user, $arr = array()){
        $tokens = array(
            'site.name'  => get_bloginfo( 'name' ),
            'site.url'   => home_url(),
            'login.url'  => wp_login_url(),
            'user.name'  => $user->user_login,
            'user.email' => $user->user_email
        );
        $tokens = wp_parse_args($tokens, $arr);
        foreach ($tokens as $key => $value) {
            $str = str_replace('{'.$key.'}', $value, $str);
        }
        return $str;
    }

    public function user_register( $user_id ) {
        if ( ! empty( $_POST[ 'pwd' ] ) ) {
            add_filter( 'send_password_change_email', '__return_false' );
            wp_update_user(
                array(
                    'ID'        => $user_id,
                    'user_pass' => wp_unslash( $_POST[ 'pwd' ] )
                )
            );
            remove_filter( 'send_password_change_email', '__return_false' );
        }

        // activate email
        if ( $this->email_activation ) {
            $user = get_userdata( $user_id );

            $code = md5( time() );
            update_user_meta( $user_id, $this->meta_key_activated, 0 );

            update_user_meta( $user_id, $this->meta_key_activate_code, $code );
            $string = array( 'id' => $user_id, 'code' => $code );
            $url    = get_site_url() . '/?act=' . base64_encode( serialize( $string ) );

            if ( $this->page_login ) {
                $url = get_permalink( $this->page_login ) . '/?act=' . base64_encode( serialize( $string ) );;
            }

            update_user_meta( $user_id, $this->meta_key_activate_link, $url );

            $should_send_email = apply_filters( 'play_block_should_send_activation_email', true, $user_id, $url, $this );
            if ( false === $should_send_email ) {
              return;
            }

            $subject = play_get_text( 'activate-email-subject' );
            $content = sprintf( play_get_text( 'activate-email-content' ), $url );

            if(play_get_option( 'email_activation_subject' )){
                $subject = $this->replace_token( play_get_option( 'email_activation_subject' ), $user, array('activation.url' => $url) );
            }

            if(play_get_option( 'email_activation_content' )){
                $content = $this->replace_token( play_get_option( 'email_activation_content' ), $user, array('activation.url' => $url) );
            }

            $mail_success = wp_mail( $user->user_email, $subject, $content );
            if ( $mail_success ) {
                play_add_message( play_get_text( 'activate-email' ), 'success' );
            } else {

            }
        } else {
            $this->autologin( $user_id );
        }
    }

    public function autologin( $user_id ) {
        if ( ! is_admin() ) {
            wp_set_current_user( $user_id );
            wp_set_auth_cookie( $user_id, true );
        }
    }

    public function verify_user_code() {
        if ( isset( $_GET[ 'act' ] ) ) {
            $data = unserialize( base64_decode( $_GET[ 'act' ] ) );
            $user_id    = isset( $data[ 'id' ] ) ? (int) $data[ 'id' ] : false;
            $data_code  = isset( $data[ 'code' ] ) ? $data[ 'code' ] : false;
            $code = get_user_meta( $user_id, $this->meta_key_activate_code, true );
            if ( $code == $data_code ) {
                update_user_meta( $user_id, $this->meta_key_activated, 1 );
                play_add_message( play_get_text( 'activated' ), 'success' );
            }
        }
    }

    public function authenticate_user( $user, $password ) {
        $activated = get_user_meta( $user->ID, $this->meta_key_activated, true );
        if ( '' === $activated ) {
            return $user;
        }
        if ( $this->email_activation && ( 1 !== (int) $activated ) ) {
            return new WP_Error( 'user_not_activated', play_get_text( 'not-activated' ) );
        }

        return $user;
    }

    public function user_row( $val, $column_name, $user_id ) {
        switch ( $column_name ) {
            case 'activate' :
                $activated = get_user_meta( $user_id, $this->meta_key_activated, true );
                if ( 1 == $activated ) {
                    return play_get_text( 'yes' );
                } else {
                    $link = get_user_meta( $user_id, $this->meta_key_activate_link, true );
                    return play_get_text( 'no' ) . '<br>' . ( $link ? sprintf( '<a href="%s">%s</a>', esc_url( $link ), play_get_text( 'activate' ) ) : '' );
                }
            case 'verified' :
                $verified = get_user_meta( $user_id, 'verified', true );
                if ( 'true' == $verified ) {
                    return play_get_text( 'yes' );
                } else {
                    return play_get_text( 'no' );
                }
            default:
        }

        return $val;
    }

    public function user_column( $columns ) {
        $columns['verified'] = play_get_text( 'verified' );

        if ( $this->email_activation ) {
            $user_activate = array(
                'activate' => play_get_text( 'activate' )
            );
            $pos           = array_search( 'email', array_keys( $columns ), true );
            $result        = array_slice( $columns, 0, $pos + 1 );
            $result        = array_merge( $result, $user_activate );

            return array_merge( $result, array_slice( $columns, $pos ) );
        }

        return $columns;
    }

    public function admin_notices() {
        remove_action( 'admin_notices', 'default_password_nag' );
    }

    public function login_form_middle() {
        ob_start();
        do_action( 'login_form' );

        return ob_get_clean();
    }

    public function register_form_middle() {
        ob_start();
        do_action( 'register_form' );

        return ob_get_clean();
    }

    public function lostpassword_form_middle() {
        ob_start();
        do_action( 'lostpassword_form' );

        return ob_get_clean();
    }

    public function login_form_top() {
        ob_start();
        do_action( 'template_notices' );

        return sprintf( '<h2>%s</h2><div class="message"></div>', __( 'Log In' ) ) . ob_get_clean();
    }

    public function login_form_bottom() {
        return $this->get_lostpwd_btn() . '  ' . $this->get_register_btn();
    }

    public function register_form_bottom() {
        return $this->get_login_btn() . '  ' . $this->get_lostpwd_btn();
    }

    public function lostpassword_form_bottom() {
        return $this->get_login_btn() . '  ' . $this->get_register_btn();
    }

    public function get_login_btn() {
        return sprintf( ' <a href="%s" data-target="#loginform" class="btn-login no-ajax">%s</a>', esc_url( wp_login_url() ), play_get_text( 'login' ) );
    }

    public function get_register_btn() {
        if ( ! $this->users_can_register ) {
            return;
        }

        return sprintf( ' <a href="%s" data-target="#registerform" class="btn-register no-ajax">%s</a>', esc_url( wp_registration_url() ), play_get_text( 'register' ) );
    }

    public function get_lostpwd_btn() {
        return sprintf( ' <a href="%s" data-target="#lostpasswordform" class="btn-lostpassword no-ajax">%s</a>', esc_url( wp_lostpassword_url() ), play_get_text( 'lost-password' ) );
    }

    public function login_form_shortcode() {
        return $this->get_login_form($_REQUEST);
    }

    public function get_login_form( $request ) {
        if ( is_user_logged_in() ) {
            return;
        }
        $action   = isset( $request[ 'action' ] ) ? sanitize_text_field( $request[ 'action' ] ) : 'login';
        $redirect = home_url();
        if ( isset( $request[ 'redirect_to' ] ) && ! empty( $request[ 'redirect_to' ] ) ) {
            $redirect = esc_url( $request[ 'redirect_to' ] );
        }
        $form = wp_login_form( array(
                'echo'     => false,
                'redirect' => $redirect
            ) ) . wp_register_form( array( 'echo' => false ) ) . wp_lostpassword_form( array( 'echo' => false ) );
        $form = str_replace( '<form ', '<form style="display:none" ', $form );
        $form = str_replace( ' style="display:none" name="' . $action . 'form"', ' name="' . $action . 'form"', $form );

        return sprintf( $this->form_wrapper, $form );
    }

    public function logged_in_redirect() {
        if ( is_user_logged_in() && $this->page_login && is_page( $this->page_login ) ) {
            wp_redirect( get_author_posts_url( get_current_user_id() ) );
            exit;
        }
    }

    public function login_url( $login_url, $redirect, $force_reauth ) {
        if ( $this->page_login ) {
            $login_url = add_query_arg( 'redirect_to', $redirect, get_permalink( $this->page_login ) );
        }

        return apply_filters( 'play_login_url', $login_url, $redirect, $force_reauth );
    }

    public function update_profile() {
        if ( ! isset( $_REQUEST[ 'update-profile-nonce' ] ) || ! isset( $_POST[ 'action' ] ) || 'update-profile' !== $_POST[ 'action' ] ) {
            return;
        }

        if ( ! wp_verify_nonce( $_REQUEST[ 'update-profile-nonce' ], 'update-profile' ) ) {
            return;
        }

        $user_id = get_current_user_id();
        if ( $user_id <= 0 ) {
            return;
        }

        $args = $this->get_profile_form_defaults();

        $current_user = get_user_by( 'id', $user_id );

        $first_name   = ! empty( $_POST[ 'first_name' ] ) ? sanitize_text_field( wp_unslash( $_POST[ 'first_name' ] ) ) : '';
        $last_name    = ! empty( $_POST[ 'last_name' ] ) ? sanitize_text_field( wp_unslash( $_POST[ 'last_name' ] ) ) : '';
        $display_name = ! empty( $_POST[ 'display_name' ] ) ? sanitize_text_field( wp_unslash( $_POST[ 'display_name' ] ) ) : '';
        $description  = ! empty( $_POST[ 'description' ] ) ? wp_kses( $_POST[ 'description' ], array(
                'br'     => array(),
                'em'     => array(),
                'strong' => array(),
                'small'  => array(),
                'span'   => array(),
                'ul'     => array(),
                'li'     => array(),
                'ol'     => array(),
                'p'      => array(),
                'a'      => array(
                    'href' => array(),
                )
            )
        ) : '';
        $url          = ! empty( $_POST[ 'url' ] ) ? sanitize_text_field( wp_unslash( $_POST[ 'url' ] ) ) : '';
        $email        = ! empty( $_POST[ 'email' ] ) ? sanitize_text_field( wp_unslash( $_POST[ 'email' ] ) ) : '';
        $pass0        = ! empty( $_POST[ 'pass' ] ) ? wp_unslash( $_POST[ 'pass' ] ) : '';
        $pass1        = ! empty( $_POST[ 'pass1' ] ) ? wp_unslash( $_POST[ 'pass1' ] ) : '';
        $pass2        = ! empty( $_POST[ 'pass2' ] ) ? wp_unslash( $_POST[ 'pass2' ] ) : '';

        $facebook  = ! empty( $_POST[ 'facebook' ] ) ? sanitize_text_field( wp_unslash( $_POST[ 'facebook' ] ) ) : '';
        $twitter   = ! empty( $_POST[ 'twitter' ] ) ? sanitize_text_field( wp_unslash( $_POST[ 'twitter' ] ) ) : '';
        $youtube   = ! empty( $_POST[ 'youtube' ] ) ? sanitize_text_field( wp_unslash( $_POST[ 'youtube' ] ) ) : '';
        $instagram = ! empty( $_POST[ 'instagram' ] ) ? sanitize_text_field( wp_unslash( $_POST[ 'instagram' ] ) ) : '';
        $snapchat  = ! empty( $_POST[ 'snapchat' ] ) ? sanitize_text_field( wp_unslash( $_POST[ 'snapchat' ] ) ) : '';
        $whatsapp  = ! empty( $_POST[ 'whatsapp' ] ) ? sanitize_text_field( wp_unslash( $_POST[ 'whatsapp' ] ) ) : '';

        $thumbnail_pos_y = ! empty( $_POST[ 'thumbnail_pos_y' ] ) ? absint( wp_unslash( $_POST[ 'thumbnail_pos_y' ] ) ) : 50;

        $user     = new stdClass();
        $user->ID = $user_id;

        $pass = true;
        /* Update user password. */
        if ( ! empty( $pass0 ) && ! empty( $pass1 ) && ! empty( $pass2 ) ) {
            if ( wp_check_password( $pass0, $current_user->user_pass, $current_user->ID ) ) {
                if ( $pass1 == $pass2 ) {
                    $user->user_pass = $pass1;
                } else {
                    $pass = false;
                    play_add_message( $args[ 'label_error_mismatch' ], 'error' );
                }
            } else {
                $pass = false;
                play_add_message( $args[ 'label_error_pass' ], 'error' );
            }
        }

        /* Update user information. */
        if ( ! empty( $email ) ) {
            if ( ! is_email( $email ) ) {
                $pass = false;
                play_add_message( $args[ 'label_error_email' ], 'error' );
            } elseif ( email_exists( $email ) && $email !== $current_user->user_email ) {
                $pass = false;
                play_add_message( $args[ 'label_error_email_exist' ], 'error' );
            } else {
                $user->user_email = $email;
            }
        }

        if ( ! empty( $first_name ) ) {
            $user->first_name = $first_name;
        }
        if ( ! empty( $last_name ) ) {
            $user->last_name = $last_name;
        }
        if ( ! empty( $display_name ) ) {
            $user->display_name = $display_name;
        }
        if ( ! empty( $description ) ) {
            $user->description = $description;
        }

        $user->user_url = esc_url( $url );
        update_user_meta( $user->ID, 'facebook', $facebook );
        update_user_meta( $user->ID, 'twitter', $twitter );
        update_user_meta( $user->ID, 'youtube', $youtube );
        update_user_meta( $user->ID, 'instagram', $instagram );
        update_user_meta( $user->ID, 'snapchat', $snapchat );
        update_user_meta( $user->ID, 'whatsapp', $whatsapp );
        update_user_meta( $user->ID, 'thumbnail_pos_y', $thumbnail_pos_y );

        do_action( 'save_user_avatar', $user->ID, 9 );

        do_action( 'profile_form_before_update', $user->ID );
        if ( $pass ) {
            wp_update_user( $user );
            do_action( 'profile_form_update', $user->ID );
            play_add_message( $args[ 'label_updated' ] );
            wp_safe_redirect( $_POST[ 'redirect_to' ] );
            exit();
        }
        do_action( 'profile_form_after_update', $user->ID );
        $url = ( is_ssl() ? 'https://' : 'http://' ) . $_SERVER[ 'HTTP_HOST' ] . $_SERVER[ 'REQUEST_URI' ];
        wp_redirect( $url );
        exit();
    }

    public function get_profile_form_defaults() {
        $defaults = array(
            'label_updated'           => play_get_text( 'saved' ),
            'label_error_email'       => play_get_text( 'error-mail' ),
            'label_error_email_exist' => play_get_text( 'error-mail-exist' ),
            'label_error_mismatch'    => play_get_text( 'error-pwd-mismatch' ),
            'label_error_pass'        => play_get_text( 'error-pwd' )
        );

        return apply_filters( 'profle_form_defaults', $defaults );
    }

    public function wp_login_form( $args = array() ) {
        return wp_login_form( $args );
    }

    public function wp_login_form_shortcode( $atts = [], $content = null, $tag = '' ) {
        $atts = array_change_key_case( (array) $atts, CASE_LOWER );
        $atts['echo'] = false;
        return sprintf( $this->form_wrapper, $this->wp_login_form( $atts ) );
    }

    public function wp_register_form_shortcode( $atts = [], $content = null, $tag = '' ) {
        $atts = array_change_key_case( (array) $atts, CASE_LOWER );
        $atts['echo'] = false;
        return sprintf( $this->form_wrapper, $this->wp_register_form( $atts ) );
    }

    public function wp_lostpassword_form_shortcode( $atts = [], $content = null, $tag = '' ) {
        $atts = array_change_key_case( (array) $atts, CASE_LOWER );
        $atts['echo'] = false;
        return sprintf( $this->form_wrapper, $this->wp_lostpassword_form( $atts ) );
    }

    public function wp_network_register() {
        if ( ! is_multisite() ) {
            return;
        }
        $http_post = ( 'POST' === $_SERVER[ 'REQUEST_METHOD' ] );
        if ( $http_post && isset( $_POST[ 'user_login' ] ) && isset( $_POST[ 'user_email' ] ) ) {
            $user_login = wp_unslash( $_POST[ 'user_login' ] );
            $user_email = wp_unslash( $_POST[ 'user_email' ] );

            $wp_error = register_new_user( $user_login, $user_email );

            if ( ! is_wp_error( $wp_error ) ) {
                $user_id = $wp_error;

                echo sprintf( '<p class="message">%s</p>', apply_filters( 'register_complete', play_get_text( 'register-complete' ) ) );

            } else {
                if ( $wp_error->has_errors() ) {
                    $errors   = '';
                    $messages = '';
                    foreach ( $wp_error->get_error_codes() as $code ) {
                        $severity = $wp_error->get_error_data( $code );
                        foreach ( $wp_error->get_error_messages( $code ) as $error_message ) {
                            if ( 'message' === $severity ) {
                                $messages .= '  ' . $error_message . "<br />\n";
                            } else {
                                $errors .= '    ' . $error_message . "<br />\n";
                            }
                        }
                    }

                    if ( ! empty( $errors ) ) {
                        echo '<div id="login_error">' . apply_filters( 'login_errors', $errors ) . "</div>\n";
                    }

                    if ( ! empty( $messages ) ) {
                        echo '<p class="message">' . apply_filters( 'login_messages', $messages ) . "</p>\n";
                    }
                }
            }
        }
    }

    public function wp_register_form( $args = array() ) {
        if ( ! $this->users_can_register ) {
            return;
        }

        // not post to wp-login.php when register on multisite
        $url = is_multisite() ? get_permalink( $this->page_login ) : site_url( 'wp-login.php?action=register', 'login_post' );

        $this->wp_network_register();

        $defaults = array(
            'echo'           => true,
            'redirect'       => home_url(),
            'form_id'        => 'registerform',
            'label_username' => __( 'Username' ),
            'label_email'    => __( 'Email' ),
            'label_password' => __( 'Password' ),
            'label_register' => __( 'Register' ),
            'title'          => __( 'Register' )
        );

        // no direct when email verification or use multisite
        if ( $this->email_activation ) {
            $defaults[ 'redirect' ] = '';
        }

        if ( is_multisite() ) {
            $defaults[ 'redirect' ] = $url;
        }

        $args = wp_parse_args( $args, apply_filters( 'register_form_defaults', $defaults ) );

        $register_form_top    = apply_filters( 'register_form_top', '<h2>' . esc_html( $args[ 'title' ] ) . '</h2>', $args );
        $register_form_middle = apply_filters( 'register_form_middle', '', $args );
        $register_form_bottom = apply_filters( 'register_form_bottom', '', $args );
        $form                 = '
            <form name="' . esc_attr( $args[ 'form_id' ] ) . '" id="' . esc_attr( $args[ 'form_id' ] ) . '" action="' . esc_url( $url ) . '" method="post">
                ' . $register_form_top . '
                <div class="message"></div>
                <p>
                    <label>' . esc_html( $args[ 'label_username' ] ) . '</label>
                    <input type="text" name="user_login" class="input" value="" required />
                </p>
                <p>
                    <label>' . esc_html( $args[ 'label_email' ] ) . '</label>
                    <input type="email" name="user_email" class="input" required />
                </p>
                <p>
                    <label>' . esc_html( $args[ 'label_password' ] ) . '</label>
                    <input type="password" name="pwd" class="input" value="" required autocomplete="off" />
                </p>
                ' . $register_form_middle . '
                <p>
                    <input type="submit" name="wp-submit" class="button button-primary" value="' . esc_attr( $args[ 'label_register' ] ) . '" />
                    <input type="hidden" name="redirect_to" value="' . esc_url( $args[ 'redirect' ] ) . '" />
                </p>
                ' . $register_form_bottom . '
            </form>';

        if ( $args[ 'echo' ] ) {
            echo $form;
        } else {
            return $form;
        }
    }

    public function wp_lostpassword_form( $args = array() ) {
        $defaults = array(
            'echo'           => true,
            'redirect'       => '',
            'form_id'        => 'lostpasswordform',
            'label_username' => __( 'Username or Email Address' ),
            'label_reset'    => __( 'Get New Password' ),
            'label_message'  => __( 'لطفا نام کاربری یا آدرس ایمیل خود را وارد نمایید. پیوندی برای ایجاد گذرواژه جدید از طریق ایمیل دریافت خواهید کرد.' ),
            'title'          => __( 'Lost Password' )
        );

        $args = wp_parse_args( $args, apply_filters( 'lostpassword_form_defaults', $defaults ) );

        $lostpassword_form_top    = apply_filters( 'lostpassword_form_top', '<h2>' . esc_html( $args[ 'title' ] ) . '</h2>', $args );
        $lostpassword_form_middle = apply_filters( 'lostpassword_form_middle', '', $args );
        $lostpassword_form_bottom = apply_filters( 'lostpassword_form_bottom', '', $args );

        $form = '
            <form name="' . esc_attr( $args[ 'form_id' ] ) . '" id="' . esc_attr( $args[ 'form_id' ] ) . '" action="' . esc_url( site_url( 'wp-login.php?action=lostpassword', 'login_post' ) ) . '" method="post">
                ' . $lostpassword_form_top . '
                <div class="message"></div>
                <p>' . esc_html( $args[ 'label_message' ] ) . '</p>
                <p>
                    <label>' . esc_html( $args[ 'label_username' ] ) . '</label>
                    <input type="text" name="user_login" class="input" required />
                </p>
                ' . $lostpassword_form_middle . '
                <p>
                    <input type="submit" name="wp-submit" class="button button-primary" value="' . esc_attr( $args[ 'label_reset' ] ) . '" />
                    <input type="hidden" name="redirect_to" value="' . esc_url( $args[ 'redirect' ] ) . '" />
                </p>
                ' . $lostpassword_form_bottom . '
            </form>';

        if ( $args[ 'echo' ] ) {
            echo $form;
        } else {
            return $form;
        }
    }

    public function wp_profile_form( $user_id ) {

        $args = $this->get_profile_form_defaults();

        if ( ! is_user_logged_in() ) {
            return;
        }
        $user = wp_get_current_user();
        if ( $user_id !== $user->ID ) {
            return;
        }

        play_get_template( 'form/profile.php', array(
                'user'     => $user,
                'redirect' => get_author_posts_url( $user->ID )
            )
        );
    }
}

Play_User_Profile::instance();
